(*  Copyright (c) 2001 Anthony L Shipman *)


signature COUNTER =
sig
    type Counter

    val new:	int -> Counter 
    val incr:	Counter -> int -> unit
    val get:	Counter -> int

end



structure Counter: COUNTER =
struct

    datatype Request = 
	    ReqIsIncr of int
	|   ReqIsGet

    and Reply =
	    ReplyIsCount of int

    and Counter = Counter of {
    		    req_chan:	Request CML.chan,
    		    rpl_chan:	Reply CML.chan
		    }


    fun new init =
    let
	val req_chan = CML.channel()
	val rpl_chan = CML.channel()

	fun counter() =
	let
	    fun loop count =
	    (
		case CML.recv req_chan of
		  ReqIsIncr n => loop (count + n)

		| ReqIsGet => 
		(
		    CML.send(rpl_chan, ReplyIsCount count);
		    loop count
		)
	    )
	in
	    loop init
	end

	val thread = CML.spawn counter
    in
	Counter
	{
	    req_chan = req_chan,
	    rpl_chan = rpl_chan
	}
    end


    fun incr (Counter {req_chan, ...}) n =
    (
	CML.send(req_chan, ReqIsIncr n)
    )


    fun get  (Counter {req_chan, rpl_chan}) =
    (
	CML.send(req_chan, ReqIsGet);

	case CML.recv rpl_chan of
	  ReplyIsCount n => n
    )

end






structure Main =
struct
    fun toErr msg = TextIO.output(TextIO.stdErr, msg)


    fun run() =
    let
	val obj = Counter.new 0
    in
	Counter.incr obj 3;
	Counter.incr obj ~1;
	print(concat["The counter's value is ", Int.toString(Counter.get obj), "\n"])
    end



    fun main(arg0, argv) =
    let
    in
	RunCML.doit(run, NONE);
        OS.Process.success
    end
    handle
      x =>
    (
	toErr(concat["Uncaught exception: ", exnMessage x, " from\n"]);
	app (fn s => (print "\t"; print s; print "\n")) (SMLofNJ.exnHistory x);
	OS.Process.failure
    )

    val _ = SMLofNJ.exportFn("counter2", main)
end



