(*  Copyright (c) 2001 Anthony L Shipman *)


(*  This measures some channel communication times.

*)

structure Main =
struct
    fun toErr msg = TextIO.output(TextIO.stdErr, msg)

    (*	The number of thousands of threads to spawn. *)
    val M = 5

    fun toTime n = Time.fromSeconds(LargeInt.fromInt n)

    (*----------------------------------------------------------*)

    datatype Msg = Msg of Time.time	(* time when the send started *)


    (*	This records a list of receiver data so that the receiver can
	run as fast as possible.
    *)

    type RxEntry = int * Time.time * Time.time

    val rx_data: RxEntry list ref = ref []

    fun report_rx() =
    let
	fun f (n, at, diff) =
	let
	in
	    print(concat["Pair ", Int.toString n, " receives at ",
		    Time.fmt 6 at, 
		    " after ",
		    LargeInt.toString(Time.toMicroseconds diff), "\n"])
	end
    in
	app f (rev(!rx_data))
    end

    (*----------------------------------------------------------*)

    fun receiver n chan () =
    let
	fun report (Msg start) =
	let
	    val diff = Time.-(Time.now(), start)
	in
	    rx_data := (n, start, diff) :: (!rx_data)
	end
    in
	(* CML.sync(CML.wrap(CML.recvEvt chan, report)) *)
	report(CML.recv chan)
    end
    handle x => toErr(concat["Uncaught exception: ", exnMessage x,
    			     " from thread ", Int.toString n, "\n"])



    (*	A message on the trigger channel starts the send on chan. *)
    fun sender n delay chan () =
    let
	fun report() = print(concat["Pair ", Int.toString n, " finishes\n"])
    in
	CML.sync(delay);
	CML.send(chan, Msg(Time.now()))
    end



    (*	This models the behaviour of the web server which creates a time-out
	event and then spawns a thread to wait for it.
    *)
    fun spawner n =
    let
	(*  Make a list of pairs of id and channel. *)
	val pairs = List.tabulate(n, fn n => (n, CML.channel()))

	(*  One timeout event will release all waiters. *)
	val delay = CML.atTimeEvt(Time.+(Time.now(), toTime 5))

	(*  Start a receiver and sender on each channel. *)
	fun startRx (n, ch) = 
	let
	    val rname = concat["Rx ", Int.toString n]
	    val sname = concat["Sn ", Int.toString n]
	in
	    Timing.timeIt rname (fn() => ignore(CML.spawn (receiver n ch)));
	    Timing.timeIt sname (fn() => ignore(CML.spawn (sender n delay ch)));
	    ()
	end

    in
	app startRx pairs
    end


    fun run argv () =
    let
	val (num, a1) = 
	    case argv of
	      []       => (20, [])
	    | (arg::r) => (valOf(Int.fromString arg), r)

	fun finish when = 
	let
	in
	    report_rx();
	    Timing.report()
	end
    in
	ignore(RunCML.addCleaner("report", [RunCML.AtShutdown], finish));
	spawner num
    end



    fun main(arg0, argv) =
    let
    in
	RunCML.doit(run argv, NONE);
        OS.Process.success
    end
    handle
      x =>
    (
	toErr(concat["Uncaught exception: ", exnMessage x, " from\n"]);
	app (fn s => (print "\t"; print s; print "\n")) (SMLofNJ.exnHistory x);
	OS.Process.failure
    )

    val _ = SMLofNJ.exportFn("chan_scaling", main)
end


