(*  Copyright (c) 2001 Anthony L Shipman *)

(* globals.sml,v 1.20 1998/03/26 00:54:28 gerald Exp *)

(*
	These are a few global flags which are typically set from the command
	line.
*)


signature GLOBALS =
sig
    val version: string
    val cgi_version: string

    val set_testing:	string -> unit
    val testing: 	int -> bool

    (*  Test flags. *)

    val TestConnect:	    int		(* show connection activity *)
    val TestTmpFile:	    int		(* show temp file activity *)
    val TestShowRequest:    int		(* show headers of a request *)
    val TestShowResponse:   int		(* show content of a response *)
    val TestStoreBuild:	    int		(* show building of the store *)
    val TestStoreProto:	    int		(* show messages in the store *)
    val TestCGIProto:	    int		(* show CGI activity in the store *)
    val TestOpenMgr:	    int		(* show OpenMgr activity *)
    val TestFinalise:	    int		(* show finaliser activity *)
    val TestAuth:	    int		(* show authorisation activity *)

    val TestProfile:	    int		(* profile the code if compiled in *)
    val TestTiming:	    int		(* log timing messages *)

    val TestTimeout:	    int		(* show timeout activity *)

end



structure Globals: GLOBALS =
struct
    open Common

    val version = "@(#)swerve 0.1"
    val cgi_version = "Swerve/0.1"	(* advertise to CGI *)

(*-----------------------------------------------------------------*)

    (*	Command line testing flags. 
	For maximum speed I do unsafe indexing into an array.
    *)

    val TestConnect		= 0
    val TestTmpFile		= 1
    val TestShowRequest		= 2
    val TestShowResponse	= 3
    val TestStoreBuild		= 4
    val TestStoreProto		= 5
    val TestCGIProto		= 6
    val TestOpenMgr		= 7
    val TestFinalise		= 8
    val TestAuth		= 9
    val TestProfile		= 10
    val TestTiming		= 11
    val TestTimeout		= 12
    val TestMAXCODE		= 200


    fun code_of_test "TMPFILE"			= TestTmpFile
    |   code_of_test "SHOWREQUEST"		= TestShowRequest
    |   code_of_test "SHOWRESPONSE"		= TestShowResponse
    |   code_of_test "STOREBUILD"		= TestStoreBuild
    |   code_of_test "STOREPROTO"		= TestStoreProto
    |   code_of_test "CGIPROTO"			= TestCGIProto
    |   code_of_test "CONNECT"			= TestConnect
    |   code_of_test "OPENMGR"			= TestOpenMgr
    |   code_of_test "FINALISE"			= TestFinalise
    |   code_of_test "AUTH"			= TestAuth
    |   code_of_test "PROFILE"			= TestProfile
    |   code_of_test "TIMING"			= TestTiming
    |   code_of_test "TIMEOUT"			= TestTimeout
    |   code_of_test _				= 0


    val zero = #"0"

    val test_flags =
	let
	    val f = Unsafe.CharArray.create (TestMAXCODE+1)
	    fun fill n = (Unsafe.CharArray.update(f, n, zero);
	    		  if n = 0 then () else fill (n-1))
	in
	    fill TestMAXCODE;
	    f
	end


    fun set_testing(flag: string) =
    let
	val code = code_of_test(upperCase flag)
    in
	if code = 0
	then
	    toErr(concat["Unrecognised test flag ", flag, "\n"])
	else
	    Unsafe.CharArray.update(test_flags, code, #"1")
    end


    fun testing query = Unsafe.CharArray.sub(test_flags, query) <> zero

(*-----------------------------------------------------------------*)

(*-----------------------------------------------------------------*)

end
