(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: signals.sml,v 1.5 2001/09/24 20:49:53 felix Exp $ *)

(*  This manages UNIX signals.

    We provide two multicast channels. The first will carry SIGINT
    and SIGTERM signals. The second will carry the sigGC signal that
    indicates a GC has happened.  The open file manager wants GC signals.

@#34567890123456789012345678901234567890123456789012345678901234567890
*)

signature SIGNAL_MGR =
sig

    (*	Each client must have its own port. *)
    type GcPort
    type IntPort

    datatype Interrupt = SIGINT | SIGTERM

    (*	This sets up the signal handling. *)
    val init:	unit -> unit


    (*	Create a new client port.
    *)
    val mkGcPort:   unit -> GcPort
    val mkIntPort:  unit -> IntPort


    (*	This creates an event for the arrival of the
	next GC signal. Call it anew for each GC.
    *)
    val gcEvt:	GcPort -> unit CML.event


    (*	This creates an event for the arrival of the
	next interrupting signal.
    *)
    val intEvt:	IntPort -> Interrupt CML.event

end



structure SignalMgr: SIGNAL_MGR =
struct
    open Common

    structure TF  = TextFrag
    structure Sig = Signals

(*------------------------------------------------------------------------------*)

    datatype Interrupt = SIGINT | SIGTERM

    type GcPort  = unit Multicast.port
    type IntPort = Interrupt Multicast.port

    val gc_mchan: unit Multicast.mchan option ref = ref NONE
    val int_mchan: Interrupt Multicast.mchan option ref = ref NONE


    fun init() =
    (
	gc_mchan  := SOME(Multicast.mChannel());
	int_mchan := SOME(Multicast.mChannel());

	Sig.setHandler(Sig.sigGC, Sig.HANDLER gc_handler);
	Sig.setHandler(Sig.sigINT, Sig.HANDLER int_handler);
	Sig.setHandler(Sig.sigTERM, Sig.HANDLER int_handler);

	(*  We'd better catch this for when writing to sockets. *)
	let
	    val s = valOf(Sig.fromString "PIPE")
	in
	    Sig.setHandler(s, Sig.HANDLER pipe_handler)
	end;
	()
    )


    and gc_handler(_, _, kont) =
    (
	Log.testInform Globals.TestTiming Log.Debug
	    (fn()=>TF.S "GC signalled");
	Multicast.multicast(valOf(!gc_mchan), ());
	kont
    )


    and int_handler(signal, _, kont) =
    let
	val msg =
	    if signal = Sig.sigINT
	    then
		SIGINT
	    else
	    if signal = Sig.sigTERM
	    then
		SIGTERM
	    else
	    (
		Log.error ["Unexpected signal in SignalMgr"];
		SIGTERM
	    )
    in
	(* REVISIT - shutdown until we get a real use for this. *)
	fail();
	Multicast.multicast(valOf(!int_mchan), msg);
	kont
    end


    and pipe_handler(signal, _, kont) =
    let
    in
	(*Log.error ["SIGPIPE in SignalMgr"];*)
	kont
    end


    fun mkGcPort() = Multicast.port(valOf(!gc_mchan))
    fun mkIntPort() = Multicast.port(valOf(!int_mchan))

    fun gcEvt port  = Multicast.recvEvt port
    fun intEvt port = Multicast.recvEvt port

(*------------------------------------------------------------------------------*)


end
