(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: singleton.sml,v 1.5 2002/03/10 17:18:02 felix Exp $ *)

(*  This is a pattern for singleton objects that are implemented
    as threads that receive a message stream from a channel or mailbox.

    For example when using a channel

	type CtrMsg = ...
	fun server ... 

	structure Counter = Singleton(
				type input    = CtrMsg CML.chan
				val  newInput = CML.channel
				val  object   = server
				)

    If the object terminates it won't be restarted. All attempts to
    communicate with it will hang.

*)

signature SINGLETON =
sig
    type input

    (*	This function returns the port into the
	singleton object.
    *)
    val get: unit -> input
end


functor Singleton (
    type input
    val  newInput: unit -> input
    val  object: input -> unit -> unit
    )
    : SINGLETON =
struct
    structure SV = SyncVar

    type input = input

    val	input: input option ref = ref NONE

    (*	An initialised mvar can be saved over an exportML.
	The value it contains is the baton, like a binary semaphore.
    *)
    val mutex = Mutex.create()


    (*	The double-checked locking will be safe in CML since it
	isn't really multi-tasking or SMP (cf Java).
    *)
    fun get() =
    (
	case !input of
	  NONE =>
	    let
		fun init() =
		(
		    case !input of
		      NONE =>
			let
			    val i = newInput()
			in
			    input := SOME i;
			    ignore(CML.spawn (object i));
			    i
			end
		    | SOME i => i
		)
	    in
		Mutex.lock mutex init
	    end

	| SOME i => i
    )
end
