(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: entity.sml,v 1.19 2002/03/10 17:18:25 felix Exp $ *)

(*  This describes a HTTP entity that is passed around and stored.

    An entity has header information: type and encoding.  It has producers
    and consumers. It can be generated lazily by a producer so the length
    may not be known by the producer. The consumer will have to count
    the size if it cares.

    A file on disk will be represented as a producer of an entity so
    there will be a need for concurrent production to different consumers.

@#345678901234567890123456789012345678901234567890123456789012345
*)

signature ENTITY =
sig


    datatype MType =
	    MType of {
		mtype:	    string,
		msubtype:   string,
		mparams:    (string * string) list
		}

	|   MTypeUnknown

    val formatType:	MType -> TextFrag.Text

    (*	This creates a simple type e.g. text/plain.
    *)
    val simpleType:	string -> string -> MType

    (*	This works out a Mime type for a file. It only
	looks at the file name.
    *)
    val getMimeType:	string -> MType

    	
    (*	This includes the HTTP1.1 content-coding.
    *)
    datatype Encoding = 
	    EncNone
	|   EncGZip
	|   EncCompress
	|   EncDeflate
	|   EncIdentity
	|   EncOther of string

    val formatEncoding:	Encoding -> string


    datatype Entity =
	    Entity of {
		info:	Info,
		body:	MKProducer
		}
	|   None

    and Info = Info of {
    	    etype:	MType option,
	    encoding:	Encoding option,
	    length:	int option,
	    last_mod:	Date.date option
	    }


    (*	A producer sends messages of this type to its consumer. *)
    and XferProto = 
	    XferInfo  of Info	    	    (* send this first *)
	|   XferBytes of Word8Vector.vector (* then lots of these *)
	|   XferDone			    (* then one of these *)
	|   XferAbort			    (* or else one of these *)

    (*	The MKProducer function must start a thread that sends the entity
	info and body to the consumer. The entity body is local to the
	producer function.  

	The length is recalculated from the frag or file.  If a file
	cannot be read then an error is logged and an empty entity
	is sent.

	The event will be enabled if the transfer is to be aborted.
	The XferAbort will be sent instead of the XferDone in this case.

	A thread id is returned so that the caller can wait until the producer
	has finished.
    *)
    withtype Consumer = XferProto CML.chan
         and MKProducer = Abort.Abort -> Info -> Consumer -> CML.thread_id

    (*	This creates a producer for an entity. *)
    val startProducer:	Abort.Abort -> Entity -> Consumer -> CML.thread_id

    (*	This is useful for a file producer which will fill in all of
	the fields from the file.
    *)
    val emptyInfo:	Info

    (*	These make producers for particular kinds of sources. 
	The lines of fragments are considered to be separated by CRLF.
	The length of the entity is defined by the body. The info length
	is only used in the xfer protocol.
    *)
    val textProducer:	TextFrag.Text -> MKProducer
    val tmpProducer:	TmpFile.TmpFile -> MKProducer
    val fileProducer:	string -> MKProducer

    (*	Beware that process producers are one-shot. 
	The holder is closed after the entity has been produced.
    *)
    val procProducer:	ExecReader.Holder -> MKProducer

end


structure Entity: ENTITY =
struct
    open Common

    structure TF = TextFrag
    structure IETF = IETF_Line

(*------------------------------------------------------------------------------*)


    datatype MType =
	    MType of {
		mtype:	    string,
		msubtype:   string,
		mparams:    (string * string) list
		}

	|   MTypeUnknown

    	
    (*	This includes the HTTP1.1 content-coding.
    *)
    datatype Encoding = 
	    EncNone
	|   EncGZip
	|   EncCompress
	|   EncDeflate
	|   EncIdentity
	|   EncOther of string


    datatype Entity =
	    Entity of {
		info:	Info,
		body:	MKProducer
		}
	|   None

    and Info = Info of {
    	    etype:	MType option,
	    encoding:	Encoding option,
	    length:	int option,
	    last_mod:	Date.date option
	    }

    (*	A producer sends messages of this type to its consumer.
    *)
    and XferProto = 
	    XferInfo  of Info	    	    (* send this first *)
	|   XferBytes of Word8Vector.vector (* then lots of these *)
	|   XferDone			    (* then one of these *)
	|   XferAbort			    (* or else one of these *)

    (*	The MKProducer function must start a thread that sends the entity
	info and body to the consumer. The entity body is local to the
	producer function.  The length should be defined by the producer
	rather than the entity info.
    *)
    withtype Consumer = XferProto CML.chan
	and  MKProducer = Abort.Abort -> Info -> Consumer -> CML.thread_id


    (*	For entity transfers. *)
    val file_chunk = 8192
    val pipe_chunk = 4096

    val emptyInfo = Info {
		etype	    = NONE,
		encoding    = NONE,
		length      = NONE,
		last_mod    = NONE
		}

(*------------------------------------------------------------------------------*)

    fun formatType (MType {mtype, msubtype, mparams}) =
    let
	fun param (key, value) = TF.L [";", key, "=", value]
    in
	case mparams of
	  [] => TF.L [mtype, "/", msubtype]	(* a common case *)

	| _  => TF.C [TF.L [mtype, "/", msubtype], TF.C(map param mparams)]
    end
    |   formatType MTypeUnknown = TF.Empty


    fun simpleType maj min =
    (
	MType {
	    mtype = maj,
	    msubtype = min,
	    mparams = []
	    }
    )


    fun formatEncoding EncNone      = ""
    |   formatEncoding EncGZip      = "x-gzip"
    |   formatEncoding EncCompress  = "x-compress"
    |   formatEncoding EncDeflate   = "deflate"
    |   formatEncoding EncIdentity  = "identity"
    |   formatEncoding (EncOther s) = IETF.quoteField s


    fun update_length
	(Info {etype, encoding, length, last_mod})
	len =
    (
	Info {
    	    etype	= etype,
	    encoding	= encoding,
	    length	= SOME len,
	    last_mod	= last_mod
	    }
    )



    (*	Work out the mime type and make a header for it.

	We'll use text/plain as the default which is OK
	for most Unix files.  Apache tries harder by acting
	like the Unix 'file' command.
    *)
    and getMimeType path =
    let
    in
	case Files.splitExt path of
	  (_, SOME ext) =>
	(
	    case Config.lookupMime ext of
	      NONE            => simpleType "text" "plain"
	    | SOME (maj, min) => simpleType maj min
	)
	| _ => simpleType "text" "plain"
    end

(*------------------------------------------------------------------------------*)

    and startProducer abort (Entity {info, body}) consumer = body abort info consumer

    |   startProducer abort None consumer =
    (
	CML.spawn (fn () => CML.send(consumer, XferDone))
    )
    


    (*	This creates a producer for a TextFrag.
    *)
    and textProducer frag abort einfo consumer =
    let
	val len = TF.length TF.UseCrLf frag

	fun producer() =
	(
	    CML.send(consumer, XferInfo(update_length einfo len));
	    TF.apply TF.UseCrLf send frag;
	    CML.send(consumer, XferDone)
	)

	and send str = CML.send(consumer, XferBytes(Byte.stringToBytes str))
    in
	CML.spawn producer
    end



    and tmpProducer tmpfile = fileProducer (TmpFile.getName tmpfile)



    (*	There is no inputNEvt() available so we can only poll once
	the transfer has started.
    *)
    and fileProducer name abort old_info consumer =
    let
	fun producer() =
	let
	    (*	All of the info fields are regenerated from the
		file at the time we send it.
	    *)
	    val opt_len = FileIO.fileSize name
	    val modt = Option.map Date.fromTimeUniv (FileIO.modTime name)

	    val info = Info {
			etype	    = SOME(getMimeType name),
			encoding    = NONE,
			length      = opt_len,
			last_mod    = modt
			}
	in
	    CML.send(consumer, XferInfo info);

	    case opt_len of 	(* NONE if can't access the file *)
	      NONE     => CML.send(consumer, XferDone)
	    | SOME len => send_file()
	end


	and send_file() =
	let
	    (*	Record the open file so that it can be finalised if
		the consumer is aborted e.g. due to a connection timeout.
	    *)
	    fun loop strm =
	    (
		if Abort.aborted abort
		then
		    CML.send(consumer, XferAbort)
		else
		let
		    val chunk = BinIO.inputN(strm, file_chunk)
		in
		    if Word8Vector.length chunk = 0
		    then
			CML.send(consumer, XferDone)
		    else
		    (
			CML.send(consumer, XferBytes chunk);
			loop strm
		    )
		end
	    )
	in
	    case BinIOReader.openIt abort name of
	      NONE   => ()
	    | SOME h => (loop (BinIOReader.get h); BinIOReader.closeIt h)
	end
	handle x => (Log.logExn x; ())

    in
	CML.spawn producer
    end



    (*	The length is taken from the info.
    *)
    and procProducer (holder: ExecReader.Holder) abort einfo consumer =
    let
	val opened as (proc, _) = ExecReader.get holder
	val (strm, _) = Unix.streamsOf proc

	fun producer() =
	(
	    CML.send(consumer, XferInfo einfo);
	    send_file();
	    ExecReader.closeIt holder;
	    ()
	)


	and send_file () =
	(
	    (*  See send_file above
		PROBLEM: CML timeouts don't seem to interrupt the inputN
		operation.
	    *)
	    if Abort.aborted abort
	    then
	    (
		CML.send(consumer, XferAbort)
	    )
	    else
	    let
		val chunk = TextIO.inputN(strm, pipe_chunk)
	    in
		if chunk = ""
		then
		(
		    CML.send(consumer, XferDone)
		)
		else
		(
		    CML.send(consumer, XferBytes(Byte.stringToBytes chunk));
		    send_file()
		)
	    end
	    handle x => (Log.logExn x; ())
	)
    in
	CML.spawn producer
    end

(*------------------------------------------------------------------------------*)

end
