(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: http_msg.sml,v 1.8 2002/01/19 16:01:45 felix Exp $ *)

(*  This defines the messages between the http protocol manager
    and the store.

    The differences between v1.0 and v1.1 are mainly in the headers.

@#345678901234567890123456789012345678901234567890123456789012345
*)

signature HTTP_MSG =
sig

    datatype Request = Request of {
		method:	    Method,
		url:	    URL.URL,
		protocol:   string,	(* e.g. "HTTP/1.0" *)
		headers:    HTTPHeader.Header list,
		entity:	    Entity.Entity,

		(*  Information from the connection. *)
		port:	    int,
		client:	    NetHostDB.in_addr,

		(*  The store sends its response here. *)
		rvar:	    Response SyncVar.ivar,

		(*  This is the abort object from the connection. *)
		abort:	    Abort.Abort
		}

    and Method = GET | HEAD | POST

    and Response =
	    Response of {
    		status:	    HTTPStatus.Status,
		headers:    HTTPHeader.Header list,
		entity:	    Entity.Entity
		}


    (*	For CGI-BIN use. *)
    val methodToString:	Method -> string


    (*	This dumps debugging information about the request to the log.
    *)
    val dumpRequest:	Request -> unit

end



structure HTTPMsg: HTTP_MSG =
struct

    structure TF = TextFrag
    structure Status = HTTPStatus
    structure Hdr = HTTPHeader

(*------------------------------------------------------------------------------*)

    datatype Request = Request of {
		method:	    Method,
		url:	    URL.URL,
		protocol:   string,	(* e.g. "HTTP/1.0" *)
		headers:    HTTPHeader.Header list,
		entity:	    Entity.Entity,

		port:	    int,
		client:	    NetHostDB.in_addr,

		rvar:	    Response SyncVar.ivar,
		abort:	    Abort.Abort
		}

    and Method = GET | HEAD | POST

    and Response = 
	    Response of {
    		status:	    HTTPStatus.Status,
		headers:    HTTPHeader.Header list,
		entity:	    Entity.Entity
		}

(*------------------------------------------------------------------------------*)


    fun methodToString GET  = "GET"
    |   methodToString HEAD = "HEAD"
    |   methodToString POST = "POST"


    fun dumpRequest (Request {method, url, protocol, headers, ...}) =
    let
	(*  We omit the body for now *)

	val tf_method = TF.C [TF.S "Method: ", TF.S (methodToString method)]

	val tf_url = TF.L ["Url: ", URL.URLToString url] 
	val tf_hdrs = TF.liftSep TF.Nl Hdr.formatHeader headers
    in
	Log.inform Log.Debug (fn()=>TF.C [
		TF.S "Request is", TF.Nl,
		tf_method, TF.Nl, 
		tf_url, TF.Nl,
		TF.L ["Protocol: ", protocol], TF.Nl,
		tf_hdrs, TF.Nl])
    end

(*------------------------------------------------------------------------------*)

end
