(*  Copyright (c) 2001 Anthony L Shipman *)

(* $Id: main.sml,v 1.16 2002/01/19 16:02:00 felix Exp $ *)


(*  This is main for the swerve server.
*)



structure Main: sig end =
struct

    open Common

(*------------------------------------------------------------------------------*)

    fun run() =
    let
    in
	TraceCML.setUncaughtFn fatal_exception; 
	(* TraceCML.setTraceFile TraceCML.TraceToErr; *)

	StartUp.startup();

	(*  Make CML run our finish() function at every shutdown. *)
	ignore(RunCML.addCleaner("finish", [RunCML.AtShutdown], 
			fn _ => StartUp.finish()));

	Listener.run();
	success()		(* shouldn't get here *)
    end


    (*	This reports on uncaught exceptions out of threads. *)
    and fatal_exception(thread, x) =
    let
	fun describe() =
	(
	    case x of
	      IO.Io {name, function, cause} =>
		toErr(concat["IO Error ", name, " ", function, " ", exnMessage cause])

	    | FatalX => ()

	    | InternalError msg => toErr(concat["Internal error, ", msg])

	    (* misc exception *) 
	    | x => toErr(concat["Uncaught exception ", exnName x, ": ",exnMessage x])
	)
    in
	describe();
	toErr(concat[" in thread ", CML.tidToString thread, "\n"]);
	fail()
    end

(*------------------------------------------------------------------------------*)

    fun usage () = (
	print "Usage: swerve [options]\n";
	print "Options:\n";
	print "  -f file     : (required) specify the server configuration file\n";
	print "  -h          : print this help message\n";
	print "  -T id       : enable test messages according to the id\n";
	print "  -D level    : set the logging level to Debug immediately\n";
	()
	)




    (*	Check for command line options. 
    *)

    fun process_args(arglist: string list) : unit =
    let
	fun loop []     = run()			(* used up all the args *)
	|   loop (arg::rest) =
	(
	    case arg of 
	      "-T"	=> loop(string_value(arg, rest, Globals.set_testing))
	    | "-D"	=> (set_debugging(); loop rest)
	    | "-f"	=> loop(string_value(arg, rest, Config.processConfig))
	    | "-h"	=> (usage(); success())
	    | _    	=> (usage(); fail())
	)


	(* Take one string as an arg and return the rest. *)
	and string_value (name, [], _) = 
	(
	    toErr (concat["Option ", name, " must have a value\n"]);
	    []
	)
	|   string_value (name, arg::rest, func: string -> unit) =
	(
	    func(arg);
	    rest
	)

	(*  Set debugging messages on so that we can see stuff during config file 
	    parsing.
	*)
	and set_debugging() = Log.setLevel Log.Debug

    in
	loop arglist
    end
    handle x => fatal_exception(CML.getTid(), x)



    fun main(arg0, argv) =
    let
	fun run() =
	(
	    SignalMgr.init();	(* required by OpenMgr *)
	    process_args argv
	)
    in
	RunCML.doit(run, NONE)
    end

    val _ = SMLofNJ.exportFn("swerve", main)
end
